<?php
// book-stats.php - Get booking click statistics for admin dashboard
// Returns monthly booking statistics for the last 12 months

require_once __DIR__ . '/config.php';

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    sendError('Only GET requests are allowed', 405);
}

try {
    // Check admin authentication
    $isAdmin = false;
    if (isset($_SERVER['HTTP_AUTHORIZATION'])) {
        $token = str_replace('Bearer ', '', $_SERVER['HTTP_AUTHORIZATION']);
        $isAdmin = validateAdminToken($token);
    }
    
    if (!$isAdmin) {
        sendError('Admin authentication required', 401);
    }
    
    // Get monthly booking statistics for the last 12 months
    $query = "
        SELECT 
            DATE_FORMAT(click_date, '%Y-%m') as month,
            COUNT(*) as click_count
        FROM booking_clicks 
        WHERE click_date >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
        GROUP BY DATE_FORMAT(click_date, '%Y-%m')
        ORDER BY month ASC
    ";
    
    $result = $db->query($query);
    
    if (!$result) {
        throw new Exception("Query failed: " . $db->error);
    }
    
    // Create array with all 12 months, filling in zeros for months with no clicks
    $stats = [];
    $currentDate = new DateTime();
    
    for ($i = 11; $i >= 0; $i--) {
        $monthDate = clone $currentDate;
        $monthDate->modify("-{$i} months");
        $monthKey = $monthDate->format('Y-m');
        $monthName = $monthDate->format('M Y');
        
        $stats[$monthKey] = [
            'month' => $monthKey,
            'monthName' => $monthName,
            'clicks' => 0
        ];
    }
    
    // Fill in actual data
    while ($row = $result->fetch_assoc()) {
        if (isset($stats[$row['month']])) {
            $stats[$row['month']]['clicks'] = (int)$row['click_count'];
        }
    }
    
    // Convert to array format for frontend
    $monthlyStats = array_values($stats);
    
    // Calculate totals
    $totalClicks = array_sum(array_column($monthlyStats, 'clicks'));
    $averageClicks = $totalClicks > 0 ? round($totalClicks / count($monthlyStats), 1) : 0;
    
    http_response_code(200);
    echo json_encode([
        'monthlyStats' => $monthlyStats,
        'totalClicks' => $totalClicks,
        'averageClicks' => $averageClicks,
        'latestMonth' => $monthlyStats[count($monthlyStats) - 1] ?? null
    ]);
    
} catch (Exception $e) {
    sendError('Failed to get booking stats: ' . $e->getMessage(), 500);
} finally {
    $database->closeConnection();
}
?>
