<?php
// doctor-single.php - Handle single doctor operations (GET, PUT, DELETE)
// Used for editing and deleting individual doctors

require_once __DIR__ . '/config.php';

$method = $_SERVER['REQUEST_METHOD'];

// Extract doctor ID from URL path
$path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$pathParts = explode('/', trim($path, '/'));
$doctorId = end($pathParts);

// Handle cases where the URL might be /doctor-single.php/5 or /doctor-single.php?id=5
if (!is_numeric($doctorId)) {
    // Try to get ID from query parameter
    $doctorId = $_GET['id'] ?? null;
}

if (!is_numeric($doctorId)) {
    sendError('Invalid doctor ID', 400);
}

try {
    // Check admin authentication
    $isAdmin = false;
    if (isset($_SERVER['HTTP_AUTHORIZATION'])) {
        $token = str_replace('Bearer ', '', $_SERVER['HTTP_AUTHORIZATION']);
        $isAdmin = validateAdminToken($token);
    }
    
    if (!$isAdmin) {
        sendError('Admin authentication required', 401);
    }

    switch ($method) {
        case 'GET':
            // Get single doctor
            $query = "SELECT * FROM doctors WHERE id = ?";
            $stmt = $db->prepare($query);
            $stmt->bind_param("i", $doctorId);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($row = $result->fetch_assoc()) {
                $doctor = [
                    'id' => (int)$row['id'],
                    'name' => $row['name'],
                    'specialty' => $row['specialty'],
                    'bio' => $row['bio'] ?? '',
                    'photoUrl' => $row['photoUrl'] ?? '',
                    'available' => (bool)$row['available'],
                    'created_at' => $row['created_at']
                ];
                
                http_response_code(200);
                echo json_encode($doctor);
            } else {
                sendError('Doctor not found', 404);
            }
            
            $stmt->close();
            break;
            
        case 'PUT':
            // Update doctor
            $data = getRequestData();
            
            if (!$data) {
                sendError('No data received', 400);
            }
            
            // Check if this is a partial update (only availability)
            if (isset($data['available']) && count($data) === 1) {
                // Partial update - only update availability
                $available = (int)$data['available'];
                
                $query = "UPDATE doctors SET available = ? WHERE id = ?";
                $stmt = $db->prepare($query);
                
                if (!$stmt) {
                    throw new Exception("Prepare failed: " . $db->error);
                }
                
                $stmt->bind_param("ii", $available, $doctorId);
                
                if ($stmt->execute()) {
                    if ($stmt->affected_rows > 0) {
                        http_response_code(200);
                        echo json_encode([
                            'success' => true,
                            'message' => 'Availability updated successfully'
                        ]);
                    } else {
                        sendError('Doctor not found or no changes made', 404);
                    }
                } else {
                    throw new Exception("Execute failed: " . $stmt->error);
                }
                
                $stmt->close();
            } else {
                // Full update - validate required fields
                $requiredFields = ['name', 'specialty', 'bio'];
                foreach ($requiredFields as $field) {
                    if (!isset($data[$field]) || empty($data[$field])) {
                        sendError("Missing required field: $field", 400);
                    }
                }
                
                // Sanitize input data
                $name = $db->real_escape_string(trim($data['name']));
                $specialty = $db->real_escape_string(trim($data['specialty']));
                $bio = $db->real_escape_string(trim($data['bio']));
                $photoUrl = isset($data['photoUrl']) ? $db->real_escape_string(trim($data['photoUrl'])) : '';
                $available = isset($data['available']) ? (int)$data['available'] : 1;
                
                // Update doctor
                $query = "UPDATE doctors SET name=?, specialty=?, bio=?, photoUrl=?, available=? WHERE id=?";
                $stmt = $db->prepare($query);
                
                if (!$stmt) {
                    throw new Exception("Prepare failed: " . $db->error);
                }
                
                $stmt->bind_param("ssssii", $name, $specialty, $bio, $photoUrl, $available, $doctorId);
                
                if ($stmt->execute()) {
                    if ($stmt->affected_rows > 0) {
                        http_response_code(200);
                        echo json_encode([
                            'success' => true,
                            'message' => 'Doctor updated successfully'
                        ]);
                    } else {
                        sendError('Doctor not found or no changes made', 404);
                    }
                } else {
                    throw new Exception("Execute failed: " . $stmt->error);
                }
                
                $stmt->close();
            }
            break;
            
        case 'DELETE':
            // Delete doctor
            $query = "DELETE FROM doctors WHERE id = ?";
            $stmt = $db->prepare($query);
            $stmt->bind_param("i", $doctorId);
            
            if ($stmt->execute()) {
                if ($stmt->affected_rows > 0) {
                    http_response_code(200);
                    echo json_encode([
                        'success' => true,
                        'message' => 'Doctor deleted successfully'
                    ]);
                } else {
                    sendError('Doctor not found', 404);
                }
            } else {
                throw new Exception("Execute failed: " . $stmt->error);
            }
            
            $stmt->close();
            break;
            
        default:
            sendError('Method not allowed', 405);
            break;
    }
    
} catch (Exception $e) {
    sendError('Operation failed: ' . $e->getMessage(), 500);
} finally {
    $database->closeConnection();
}
?>
