<?php
// doctors-admin.php - Admin-specific doctor management endpoint
// Handles GET (all doctors), POST (create), PUT (update), DELETE operations

require_once __DIR__ . '/config.php';

$method = $_SERVER['REQUEST_METHOD'];

try {
    // Check admin authentication
    $isAdmin = false;
    if (isset($_SERVER['HTTP_AUTHORIZATION'])) {
        $token = str_replace('Bearer ', '', $_SERVER['HTTP_AUTHORIZATION']);
        $isAdmin = validateAdminToken($token);
    }
    
    if (!$isAdmin) {
        sendError('Admin authentication required', 401);
    }

    switch ($method) {
        case 'GET':
            // Get all doctors for admin (including unavailable ones)
            $query = "SELECT * FROM doctors ORDER BY created_at DESC";
            $result = $db->query($query);
            
            if (!$result) {
                throw new Exception("Query failed: " . $db->error);
            }
            
            $doctors = [];
            while ($row = $result->fetch_assoc()) {
                $doctors[] = [
                    'id' => (int)$row['id'],
                    'name' => $row['name'],
                    'specialty' => $row['specialty'],
                    'bio' => $row['bio'] ?? '',
                    'photoUrl' => $row['photoUrl'] ?? '',
                    'available' => (bool)$row['available'],
                    'created_at' => $row['created_at']
                ];
            }
            
            http_response_code(200);
            echo json_encode($doctors);
            break;
            
        case 'POST':
            // Create new doctor
            $data = getRequestData();
            
            if (!$data) {
                sendError('No data received', 400);
            }
            
            // Validate required fields
            $requiredFields = ['name', 'specialty', 'bio'];
            foreach ($requiredFields as $field) {
                if (!isset($data[$field]) || empty($data[$field])) {
                    sendError("Missing required field: $field", 400);
                }
            }
            
            // Sanitize input data
            $name = $db->real_escape_string(trim($data['name']));
            $specialty = $db->real_escape_string(trim($data['specialty']));
            $bio = $db->real_escape_string(trim($data['bio']));
            $photoUrl = isset($data['photoUrl']) ? $db->real_escape_string(trim($data['photoUrl'])) : '';
            $available = isset($data['available']) ? (int)$data['available'] : 1;
            
            // Insert new doctor
            $query = "INSERT INTO doctors (name, specialty, bio, photoUrl, available, created_at) VALUES (?, ?, ?, ?, ?, NOW())";
            $stmt = $db->prepare($query);
            
            if (!$stmt) {
                throw new Exception("Prepare failed: " . $db->error);
            }
            
            $stmt->bind_param("ssssi", $name, $specialty, $bio, $photoUrl, $available);
            
            if ($stmt->execute()) {
                $newId = $db->insert_id;
                
                http_response_code(201);
                echo json_encode([
                    'success' => true,
                    'id' => $newId,
                    'message' => 'Doctor created successfully'
                ]);
            } else {
                throw new Exception("Execute failed: " . $stmt->error);
            }
            
            $stmt->close();
            break;
            
        default:
            sendError('Method not allowed', 405);
            break;
    }
    
} catch (Exception $e) {
    sendError('Operation failed: ' . $e->getMessage(), 500);
} finally {
    $database->closeConnection();
}
?>
