<?php
// get_users.php - Returns doctor data with proper CORS headers
// Demonstrates GET request handling and database queries

require_once __DIR__ . '/config.php';

// Set CORS headers for HTTPS frontend
header("Access-Control-Allow-Origin: https://www.sheydoc.com");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
header("Access-Control-Allow-Credentials: true");

// Handle preflight OPTIONS requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

try {
    // Check if this is an admin request (optional)
    $isAdmin = false;
    if (isset($_SERVER['HTTP_AUTHORIZATION'])) {
        $token = str_replace('Bearer ', '', $_SERVER['HTTP_AUTHORIZATION']);
        $isAdmin = validateAdminToken($token);
    }
    
    // Check for specialty filter
    $specialty = $_GET['specialty'] ?? null;
    
    // Query to get users (you can modify this based on your actual table structure)
    if ($specialty) {
        $query = "SELECT * FROM doctors WHERE specialty LIKE ? ORDER BY created_at DESC";
        $stmt = $db->prepare($query);
        $stmt->bind_param("s", $specialty);
        $stmt->execute();
        $result = $stmt->get_result();
    } else {
        $query = "SELECT * FROM doctors ORDER BY created_at DESC";
        $result = $db->query($query);
    }
    
    if (!$result) {
        throw new Exception("Query failed: " . $db->error);
    }
    
    $users = [];
    while ($row = $result->fetch_assoc()) {
        // Only include available doctors for public requests
        if (!$isAdmin && !$row['available']) {
            continue;
        }
        
        // Sanitize data for public consumption
        $user = [
            'id' => (int)$row['id'],
            'name' => $row['name'],
            'specialty' => $row['specialty'],
            'bio' => $row['bio'] ?? '',
            'photoUrl' => $row['photoUrl'] ?? '',
            'available' => (bool)$row['available'],
            'created_at' => $row['created_at']
        ];
        
        $users[] = $user;
    }
    
    // Return doctors array directly for frontend compatibility
    http_response_code(200);
    header('Content-Type: application/json');
    echo json_encode($users);
    exit();
    
} catch (Exception $e) {
    sendError('Failed to get users: ' . $e->getMessage(), 500);
} finally {
    $database->closeConnection();
}
?>
