<?php
// save_data.php - Handles POST requests to save data
// Demonstrates POST request handling and database inserts

require_once __DIR__ . '/config.php';

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendError('Only POST requests are allowed', 405);
}

try {
    // Get request data
    $data = getRequestData();
    
    if (!$data) {
        sendError('No data received', 400);
    }
    
    // Validate required fields (customize based on your needs)
    $requiredFields = ['name', 'specialty'];
    foreach ($requiredFields as $field) {
        if (!isset($data[$field]) || empty($data[$field])) {
            sendError("Missing required field: $field", 400);
        }
    }
    
    // Sanitize input data
    $name = $db->real_escape_string(trim($data['name']));
    $specialty = $db->real_escape_string(trim($data['specialty']));
    $email = isset($data['email']) ? $db->real_escape_string(trim($data['email'])) : null;
    $phone = isset($data['phone']) ? $db->real_escape_string(trim($data['phone'])) : null;
    $available = isset($data['available']) ? (int)$data['available'] : 1;
    
    // Insert data into database
    $query = "INSERT INTO doctors (name, specialty, email, phone, available, created_at) VALUES (?, ?, ?, ?, ?, NOW())";
    $stmt = $db->prepare($query);
    
    if (!$stmt) {
        throw new Exception("Prepare failed: " . $db->error);
    }
    
    $stmt->bind_param("ssssi", $name, $specialty, $email, $phone, $available);
    
    if ($stmt->execute()) {
        $newId = $db->insert_id;
        
        sendResponse([
            'message' => 'Data saved successfully',
            'id' => $newId,
            'saved_data' => [
                'id' => $newId,
                'name' => $name,
                'specialty' => $specialty,
                'email' => $email,
                'phone' => $phone,
                'available' => (bool)$available
            ]
        ], 201);
    } else {
        throw new Exception("Execute failed: " . $stmt->error);
    }
    
    $stmt->close();
    
} catch (Exception $e) {
    sendError('Failed to save data: ' . $e->getMessage(), 500);
} finally {
    $database->closeConnection();
}
?>
