<?php
// upload-image.php - Handles image uploads for doctor profiles
// Supports multipart/form-data uploads with validation

require_once __DIR__ . '/config.php';

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendError('Only POST requests are allowed', 405);
}

// Check for admin authentication
$isAdmin = false;
if (isset($_SERVER['HTTP_AUTHORIZATION'])) {
    $token = str_replace('Bearer ', '', $_SERVER['HTTP_AUTHORIZATION']);
    $isAdmin = validateAdminToken($token);
}

if (!$isAdmin) {
    sendError('Admin authentication required', 401);
}

try {
    // Check if file was uploaded
    if (!isset($_FILES['image']) || $_FILES['image']['error'] !== UPLOAD_ERR_OK) {
        sendError('No image file uploaded or upload error occurred', 400);
    }

    $file = $_FILES['image'];
    
    // Validate file type
    $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
    $fileType = mime_content_type($file['tmp_name']);
    
    if (!in_array($fileType, $allowedTypes)) {
        sendError('Invalid file type. Only JPEG, PNG, GIF, and WebP images are allowed.', 400);
    }
    
    // Validate file size (max 5MB)
    $maxSize = 5 * 1024 * 1024; // 5MB in bytes
    if ($file['size'] > $maxSize) {
        sendError('File size too large. Maximum size is 5MB.', 400);
    }
    
    // Create uploads directory if it doesn't exist
    $uploadDir = __DIR__ . '/uploads/doctors/';
    if (!is_dir($uploadDir)) {
        if (!mkdir($uploadDir, 0755, true)) {
            throw new Exception('Failed to create upload directory');
        }
    }
    
    // Generate unique filename
    $fileExtension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = 'doctor_' . time() . '_' . uniqid() . '.' . $fileExtension;
    $uploadPath = $uploadDir . $filename;
    
    // Move uploaded file
    if (move_uploaded_file($file['tmp_name'], $uploadPath)) {
        // Generate the full URL for the uploaded image
        $imageUrl = 'https://api.sheydoc.com/uploads/doctors/' . $filename;
        
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'imageUrl' => $imageUrl,
            'filename' => $filename,
            'message' => 'Image uploaded successfully'
        ]);
        exit();
    } else {
        sendError('Failed to move uploaded file', 500);
    }
    
} catch (Exception $e) {
    sendError('Upload failed: ' . $e->getMessage(), 500);
} finally {
    $database->closeConnection();
}
?>
